<?php
require __DIR__ . '/../../vendor/autoload.php';
$app = require_once __DIR__ . '/../../bootstrap/app.php';
$kernel = $app->make(Illuminate\Contracts\Console\Kernel::class);
$kernel->bootstrap();

use App\Services\GoogleAnalyticsService;
use Carbon\Carbon;
use Illuminate\Support\Facades\DB;

// Range: last 30 days by default
$to = Carbon::now();
$from = Carbon::now()->subDays(29);

echo "Backfilling GA aggregated page views from " . $from->toDateString() . " to " . $to->toDateString() . "\n";

$svc = app(GoogleAnalyticsService::class);

// create aggregated table if not exists (MySQL compatible)
DB::statement("CREATE TABLE IF NOT EXISTS page_view_daily (
    id BIGINT PRIMARY KEY AUTO_INCREMENT,
    path VARCHAR(191) NOT NULL,
    title VARCHAR(191),
    views BIGINT DEFAULT 0,
    view_date DATE,
    created_at TIMESTAMP NULL,
    updated_at TIMESTAMP NULL,
    UNIQUE KEY pv_daily_path_date (path, view_date)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;");

$current = $from->copy();
$totalRows = 0;
$totalViews = 0;

while ($current->lte($to)) {
    $dayStart = $current->copy()->startOfDay();
    $dayEnd = $current->copy()->endOfDay();
    echo "Processing " . $dayStart->toDateString() . "...\n";

    try {
        $rows = $svc->getTopPages($dayStart, $dayEnd, 2000) ?? [];
    } catch (\Exception $e) {
        echo "GA fetch failed for {$dayStart->toDateString()}: " . $e->getMessage() . "\n";
        $rows = [];
    }

    foreach ($rows as $r) {
        $path = $r['path'] ?? null;
        if (empty($path)) continue;
        if (strpos($path, '/admin') === 0) continue; // skip admin
        $title = $r['title'] ?? $path;
        $views = (int) ($r['views'] ?? 0);

        DB::table('page_view_daily')->updateOrInsert(
            ['path' => $path, 'view_date' => $dayStart->toDateString()],
            ['title' => $title, 'views' => $views, 'created_at' => $dayStart->toDateTimeString(), 'updated_at' => now()->toDateTimeString()]
        );

        $totalRows++;
        $totalViews += $views;
    }

    $current->addDay();
}

echo "Backfill complete. Rows upserted={$totalRows} totalViews={$totalViews}\n";
