<?php
// Lightweight GA backfill script (no Laravel bootstrap)
// Usage: php application/scripts/light_ga_backfill.php --from=YYYY-MM-DD --to=YYYY-MM-DD

require __DIR__ . '/../vendor/autoload.php';

use GuzzleHttp\Client;
use Carbon\Carbon;

function parse_args()
{
    $args = $_SERVER['argv'];
    $out = [];
    foreach ($args as $arg) {
        if (strpos($arg, '--from=') === 0) $out['from'] = substr($arg, 7);
        if (strpos($arg, '--to=') === 0) $out['to'] = substr($arg, 5);
    }
    return $out;
}

function load_env_file($path)
{
    if (!file_exists($path)) return [];
    $lines = file($path, FILE_IGNORE_NEW_LINES | FILE_SKIP_EMPTY_LINES);
    $vars = [];
    foreach ($lines as $line) {
        if (strpos(trim($line), '#') === 0) continue;
        if (!strpos($line, '=')) continue;
        [$k, $v] = explode('=', $line, 2);
        $vars[trim($k)] = trim(trim($v), "\"'");
    }
    return $vars;
}

function get_env($key, $fallbacks = [])
{
    $val = getenv($key);
    if ($val !== false) return $val;
    foreach ($fallbacks as $fb) {
        if (isset($fb[$key])) return $fb[$key];
    }
    return null;
}

function get_access_token_from_service_account($credentialsSource)
{
    // credentialsSource may be a path or raw JSON or base64 JSON
    if (file_exists($credentialsSource)) {
        $json = json_decode(file_get_contents($credentialsSource), true);
    } else {
        $maybe = $credentialsSource;
        // try base64
        $decoded = base64_decode($maybe, true);
        if ($decoded !== false && strpos(ltrim($decoded), '{') === 0) $maybe = $decoded;
        $json = json_decode($maybe, true);
    }
    if (!$json || empty($json['client_email']) || empty($json['private_key'])) return null;

    $now = time();
    $iss = $json['client_email'];
    $scope = 'https://www.googleapis.com/auth/analytics.readonly';
    $aud = 'https://oauth2.googleapis.com/token';

    $jwtHeader = rtrim(strtr(base64_encode(json_encode(['alg' => 'RS256', 'typ' => 'JWT'])), '+/', '-_'), '=');
    $jwtClaim = rtrim(strtr(base64_encode(json_encode([
        'iss' => $iss,
        'scope' => $scope,
        'aud' => $aud,
        'exp' => $now + 3600,
        'iat' => $now
    ])), '+/', '-_'), '=');

    $signatureInput = $jwtHeader . '.' . $jwtClaim;
    $privateKey = openssl_pkey_get_private($json['private_key']);
    if (!$privateKey) return null;
    $signature = '';
    openssl_sign($signatureInput, $signature, $privateKey, OPENSSL_ALGO_SHA256);
    $jwtSig = rtrim(strtr(base64_encode($signature), '+/', '-_'), '=');
    $jwt = $signatureInput . '.' . $jwtSig;

    $client = new Client(['timeout' => 15]);
    try {
        $res = $client->post('https://oauth2.googleapis.com/token', [
            'form_params' => [
                'grant_type' => 'urn:ietf:params:oauth:grant-type:jwt-bearer',
                'assertion' => $jwt
            ]
        ]);
        $body = json_decode((string)$res->getBody(), true);
        return $body['access_token'] ?? null;
    } catch (\Exception $e) {
        fwrite(STDERR, "Token request failed: " . $e->getMessage() . "\n");
        return null;
    }
}

function fetch_top_pages($accessToken, $propertyId, $from, $to, $limit = 2000)
{
    $client = new Client(['timeout' => 20]);
    $body = [
        'dateRanges' => [['startDate' => $from->toDateString(), 'endDate' => $to->toDateString()]],
        'dimensions' => [['name' => 'pageTitle'], ['name' => 'pagePath'], ['name' => 'eventName']],
        'metrics' => [['name' => 'eventCount']],
        'limit' => (int)$limit
    ];
    try {
        $res = $client->post('https://analyticsdata.googleapis.com/v1beta/properties/' . $propertyId . ':runReport', [
            'headers' => [
                'Authorization' => 'Bearer ' . $accessToken,
                'Content-Type' => 'application/json'
            ],
            'body' => json_encode($body)
        ]);
        $data = json_decode((string)$res->getBody(), true);
        $out = [];
        if (!empty($data['rows'])) {
            foreach ($data['rows'] as $row) {
                $title = $row['dimensionValues'][0]['value'] ?? null;
                $path = $row['dimensionValues'][1]['value'] ?? null;
                $eventName = $row['dimensionValues'][2]['value'] ?? null;
                $views = isset($row['metricValues'][0]['value']) ? (int)$row['metricValues'][0]['value'] : 0;
                if ($eventName === 'page_view' && !empty($path)) {
                    $out[] = ['title' => $title ?? $path, 'path' => $path, 'views' => $views];
                }
            }
        }
        return $out;
    } catch (\Exception $e) {
        fwrite(STDERR, "GA fetch failed: " . $e->getMessage() . "\n");
        return [];
    }
}

function create_table_if_missing($pdo)
{
    $sql = "CREATE TABLE IF NOT EXISTS page_view_daily (
        id BIGINT PRIMARY KEY AUTO_INCREMENT,
        path VARCHAR(191) NOT NULL,
        title VARCHAR(191),
        views BIGINT DEFAULT 0,
        view_date DATE,
        created_at TIMESTAMP NULL,
        updated_at TIMESTAMP NULL,
        UNIQUE KEY pv_daily_path_date (path, view_date)
    ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;";
    $pdo->exec($sql);
}

function upsert_row($pdo, $path, $title, $views, $view_date)
{
    $now = date('Y-m-d H:i:s');
    $stmt = $pdo->prepare(
        "INSERT INTO page_view_daily (path, title, views, view_date, created_at, updated_at)
         VALUES (:path, :title, :views, :view_date, :created_at, :updated_at)
         ON DUPLICATE KEY UPDATE title = VALUES(title), views = VALUES(views), updated_at = VALUES(updated_at)"
    );
    $stmt->execute([
        ':path' => $path,
        ':title' => $title,
        ':views' => $views,
        ':view_date' => $view_date,
        ':created_at' => $now,
        ':updated_at' => $now
    ]);
}

// ---- main ----
$args = parse_args();
if (empty($args['from']) || empty($args['to'])) {
    fwrite(STDERR, "Usage: php light_ga_backfill.php --from=YYYY-MM-DD --to=YYYY-MM-DD\n");
    exit(2);
}

 $envFile = __DIR__ . '/../.env';
$envVars = load_env_file($envFile);

$credentialsSource = get_env('GA_SERVICE_ACCOUNT_JSON', [$envVars]);
$propertyId = get_env('GA_PROPERTY_ID', [$envVars]);
if (empty($credentialsSource) || empty($propertyId)) {
    fwrite(STDERR, "GA_SERVICE_ACCOUNT_JSON or GA_PROPERTY_ID not found in environment or .env\n");
    exit(2);
}

$accessToken = get_access_token_from_service_account($credentialsSource);
if (!$accessToken) {
    fwrite(STDERR, "Could not obtain GA access token\n");
    exit(2);
}

$from = Carbon::parse($args['from']);
$to = Carbon::parse($args['to']);

// DB connection via .env values
$dbHost = get_env('DB_HOST', [$envVars]) ?: '127.0.0.1';
$dbPort = get_env('DB_PORT', [$envVars]) ?: '3306';
$dbName = get_env('DB_DATABASE', [$envVars]) ?: null;
$dbUser = get_env('DB_USERNAME', [$envVars]) ?: null;
$dbPass = get_env('DB_PASSWORD', [$envVars]) ?: null;

if (!$dbName || !$dbUser) {
    fwrite(STDERR, "DB_DATABASE or DB_USERNAME missing in environment/.env\n");
    exit(2);
}

$dsn = "mysql:host={$dbHost};port={$dbPort};dbname={$dbName};charset=utf8mb4";
try {
    $pdo = new PDO($dsn, $dbUser, $dbPass, [PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION]);
} catch (\Exception $e) {
    fwrite(STDERR, "DB connection failed: " . $e->getMessage() . "\n");
    exit(2);
}

create_table_if_missing($pdo);

$current = $from->copy();
$totalRows = 0;
$totalViews = 0;

while ($current->lte($to)) {
    $dayStart = $current->copy()->startOfDay();
    $dayEnd = $current->copy()->endOfDay();
    echo "Processing " . $dayStart->toDateString() . "...\n";

    $rows = fetch_top_pages($accessToken, $propertyId, $dayStart, $dayEnd, 2000);
    foreach ($rows as $r) {
        $path = $r['path'] ?? null;
        if (empty($path)) continue;
        if (strpos($path, '/admin') === 0) continue;
        $title = $r['title'] ?? $path;
        $views = (int)($r['views'] ?? 0);
        upsert_row($pdo, $path, $title, $views, $dayStart->toDateString());
        $totalRows++;
        $totalViews += $views;
    }

    $current->addDay();
}

echo "Backfill complete. Rows upserted={$totalRows} totalViews={$totalViews}\n";

exit(0);
