#!/usr/bin/env bash
# Helper to run the GA backfill artisan command with adjustable memory and per-day looping.
# Usage:
#   ./application/scripts/ga_backfill.sh --from=2025-12-17 --to=2025-12-19
# Options:
#   --from=YYYY-MM-DD   start date (inclusive)
#   --to=YYYY-MM-DD     end date (inclusive)
#   --memory=1G         PHP memory_limit to use (default: 1G)

set -euo pipefail

FROM=""
TO=""
MEM="1G"

for arg in "$@"; do
  case $arg in
    --from=*) FROM="${arg#*=}" ;;
    --to=*) TO="${arg#*=}" ;;
    --memory=*) MEM="${arg#*=}" ;;
    *) echo "Unknown arg: $arg"; exit 2 ;;
  esac
done

if [ -z "$FROM" ] || [ -z "$TO" ]; then
  echo "Usage: $0 --from=YYYY-MM-DD --to=YYYY-MM-DD [--memory=1G]"
  exit 2
fi

echo "Running GA backfill from $FROM to $TO with PHP memory_limit=$MEM"

# Preferred: run artisan once for the full range (faster). If this fails due to memory,
# you can run with --per-day to call per-day (slower but may bypass transient memory spikes).
ARTISAN="$(pwd)/application/artisan"

if [ ! -f "$ARTISAN" ]; then
  echo "Could not find artisan at $ARTISAN. Run this script from the project root." >&2
  exit 2
fi

php -d memory_limit=$MEM "$ARTISAN" ga:backfill-page-views --from="$FROM" --to="$TO" --mode=aggregated

EXIT_CODE=$?
if [ $EXIT_CODE -eq 0 ]; then
  echo "Backfill finished successfully."
else
  echo "Backfill exited with code $EXIT_CODE" >&2
fi

echo "If you still hit PHP memory errors, try increasing --memory (e.g. 2G), or run per-day:"
echo "  for d in \\$(seq 0 2); do DAY=\$(date -j -v+${d}d -f %Y-%m-%d $FROM +%Y-%m-%d); php -d memory_limit=1G \"$ARTISAN\" ga:backfill-page-views --from=\$DAY --to=\$DAY --mode=aggregated; done"
