<?php

namespace App\Services;

use GuzzleHttp\Client;
use Carbon\Carbon;

class GoogleAnalyticsService
{
    protected $client;
    protected $credentialsSource;
    protected $propertyId;

    public function __construct()
    {
        $this->client = new Client();
        // `GA_SERVICE_ACCOUNT_JSON` may be either: an absolute file path to the JSON key,
        // or the JSON content (or base64 encoded JSON) itself (helpful for deployment envs).
        $this->credentialsSource = env('GA_SERVICE_ACCOUNT_JSON');
        $this->propertyId = env('GA_PROPERTY_ID'); // numeric id, e.g. 123456789
    }

    protected function hasConfig()
    {
        if (empty($this->credentialsSource) || empty($this->propertyId) || !env('GA_ENABLED')) {
            return false;
        }

        // If it's a path and file exists, ok. Otherwise we'll try to parse it as JSON later.
        if (file_exists($this->credentialsSource)) {
            return true;
        }

        // If the source is not a path, attempt simple JSON decode to validate.
        $decoded = json_decode($this->maybeDecodeBase64($this->credentialsSource), true);
        return is_array($decoded) && !empty($decoded['client_email']) && !empty($decoded['private_key']);
    }

    /**
     * Get an access token using service account JSON and JWT exchange
     * @return string|null
     */
    protected function getAccessToken()
    {
        if (!$this->hasConfig()) {
            return null;
        }

        // Load credentials: prefer file path, otherwise try to parse the env content.
        $json = null;
        if (file_exists($this->credentialsSource)) {
            $json = json_decode(file_get_contents($this->credentialsSource), true);
        } else {
            $maybe = $this->maybeDecodeBase64($this->credentialsSource);
            $json = json_decode($maybe, true);
        }

        if (!$json || empty($json['client_email']) || empty($json['private_key'])) {
            return null;
        }

        $now = time();
        $iss = $json['client_email'];
        $scope = 'https://www.googleapis.com/auth/analytics.readonly';
        $aud = 'https://oauth2.googleapis.com/token';

        $jwtHeader = $this->base64UrlEncode(json_encode(['alg' => 'RS256', 'typ' => 'JWT']));
        $jwtClaim = $this->base64UrlEncode(json_encode([
            'iss' => $iss,
            'scope' => $scope,
            'aud' => $aud,
            'exp' => $now + 3600,
            'iat' => $now
        ]));

        $signatureInput = $jwtHeader . '.' . $jwtClaim;
        $privateKey = openssl_pkey_get_private($json['private_key']);
        if (!$privateKey) return null;

        $signature = '';
        openssl_sign($signatureInput, $signature, $privateKey, OPENSSL_ALGO_SHA256);
        $jwtSig = $this->base64UrlEncode($signature);
        $jwt = $signatureInput . '.' . $jwtSig;

        try {
            $res = $this->client->post('https://oauth2.googleapis.com/token', [
                'form_params' => [
                    'grant_type' => 'urn:ietf:params:oauth:grant-type:jwt-bearer',
                    'assertion' => $jwt
                ],
                'timeout' => 10
            ]);

            $body = json_decode((string) $res->getBody(), true);
            return $body['access_token'] ?? null;
        } catch (\Exception $e) {
            return null;
        }
    }

    /**
     * If the provided string looks like base64, decode it; otherwise return as-is.
     * This helps support env values that carry the JSON key as base64.
     */
    protected function maybeDecodeBase64($value)
    {
        if (empty($value) || !is_string($value)) return $value;

        // strip surrounding quotes if present
        $value = trim($value, "\"'");

        // Heuristic: if it contains whitespace/newlines and starts with '{', it's raw JSON.
        if (strpos(ltrim($value), '{') === 0) {
            return $value;
        }

        // If it's likely base64 (no spaces and =~ base64 chars), try decode
        if (preg_match('%^[A-Za-z0-9+/=\r\n]+$%', $value)) {
            $decoded = base64_decode($value, true);
            if ($decoded !== false && strpos(ltrim($decoded), '{') === 0) {
                return $decoded;
            }
        }

        return $value;
    }

    protected function base64UrlEncode($data)
    {
        return rtrim(strtr(base64_encode($data), '+/', '-_'), '=');
    }

    /**
     * Request total users for a date range using Analytics Data API (GA4)
     * @param Carbon|null $from
     * @param Carbon|null $to
     * @return int|null
     */
    public function getTotalUsers($from = null, $to = null)
    {
        if (!$this->hasConfig()) {
            return null;
        }

        $accessToken = $this->getAccessToken();
        if (!$accessToken) return null;

        // default to last 30 days if no range
        if (!$from || !$to) {
            $to = Carbon::today();
            $from = Carbon::today()->subDays(30);
        }

        $startDate = $from->toDateString();
        $endDate = $to->toDateString();

        $body = [
            'dateRanges' => [
                ['startDate' => $startDate, 'endDate' => $endDate]
            ],
            'metrics' => [['name' => 'totalUsers']]
        ];

        try {
            $res = $this->client->post('https://analyticsdata.googleapis.com/v1beta/properties/' . $this->propertyId . ':runReport', [
                'headers' => [
                    'Authorization' => 'Bearer ' . $accessToken,
                    'Content-Type' => 'application/json'
                ],
                'body' => json_encode($body),
                'timeout' => 15
            ]);

            $data = json_decode((string) $res->getBody(), true);
            // when no rows, the metric might be in totals
            if (!empty($data['rows'])) {
                // sum metric values if multiple rows
                $sum = 0;
                foreach ($data['rows'] as $row) {
                    if (!empty($row['metricValues'][0]['value'])) {
                        $sum += (int) $row['metricValues'][0]['value'];
                    }
                }
                return $sum;
            }

            if (!empty($data['totals'][0]['values'][0])) {
                return (int) $data['totals'][0]['values'][0];
            }

            return null;
        } catch (\Exception $e) {
            return null;
        }
    }

    /**
     * Request page views for a date range using Analytics Data API (GA4)
     * @param Carbon|null $from
     * @param Carbon|null $to
     * @return int|null
     */
    public function getPageViews($from = null, $to = null)
    {
        if (!$this->hasConfig()) {
            return null;
        }

        $accessToken = $this->getAccessToken();
        if (!$accessToken) return null;

        // default to last 30 days if no range
        if (!$from || !$to) {
            $to = Carbon::today();
            $from = Carbon::today()->subDays(30);
        }

        $startDate = $from->toDateString();
        $endDate = $to->toDateString();

        // Request eventCount broken down by eventName, then sum only page_view rows.
        $body = [
            'dateRanges' => [ ['startDate' => $startDate, 'endDate' => $endDate] ],
            'dimensions' => [ ['name' => 'eventName'] ],
            'metrics' => [ ['name' => 'eventCount'] ],
            'limit' => 10000
        ];

        try {
            $res = $this->client->post('https://analyticsdata.googleapis.com/v1beta/properties/' . $this->propertyId . ':runReport', [
                'headers' => [
                    'Authorization' => 'Bearer ' . $accessToken,
                    'Content-Type' => 'application/json'
                ],
                'body' => json_encode($body),
                'timeout' => 15
            ]);

            $data = json_decode((string) $res->getBody(), true);
            // Sum only rows where eventName == 'page_view'
            $sum = 0;
            if (!empty($data['rows'])) {
                foreach ($data['rows'] as $row) {
                    $event = $row['dimensionValues'][0]['value'] ?? null;
                    $val = $row['metricValues'][0]['value'] ?? null;
                    if ($event === 'page_view' && $val !== null) {
                        $sum += (int) $val;
                    }
                }
            }

            // If API returned totals, prefer totals where possible (may include all events),
            // but we've already summed page_view rows above.
            if ($sum > 0) return $sum;

            if (!empty($data['totals'][0]['values'][0])) {
                return (int) $data['totals'][0]['values'][0];
            }

            return null;
        } catch (\Exception $e) {
            return null;
        }
    }

    /**
     * Request realtime active users using the Realtime API (GA4)
     * @return int|null
     */
    public function getRealtimeUsers()
    {
        if (!$this->hasConfig()) {
            return null;
        }

        $accessToken = $this->getAccessToken();
        if (!$accessToken) return null;

        $body = [
            'metrics' => [['name' => 'activeUsers']]
        ];

        try {
            $res = $this->client->post('https://analyticsdata.googleapis.com/v1beta/properties/' . $this->propertyId . ':runRealtimeReport', [
                'headers' => [
                    'Authorization' => 'Bearer ' . $accessToken,
                    'Content-Type' => 'application/json'
                ],
                'body' => json_encode($body),
                'timeout' => 15
            ]);

            $data = json_decode((string) $res->getBody(), true);

            if (!empty($data['rows'])) {
                $sum = 0;
                foreach ($data['rows'] as $row) {
                    if (!empty($row['metricValues'][0]['value'])) {
                        $sum += (int) $row['metricValues'][0]['value'];
                    }
                }
                return $sum;
            }

            if (!empty($data['totals'][0]['values'][0])) {
                return (int) $data['totals'][0]['values'][0];
            }

            return null;
        } catch (\Exception $e) {
            return null;
        }
    }

    /**
     * Request realtime pageviews (eventCount for page_view) using the Realtime API (GA4)
     * This returns the number of page_view events seen in realtime (if available).
     * @return int|null
     */
    public function getRealtimePageViews()
    {
        if (!$this->hasConfig()) {
            return null;
        }

        $accessToken = $this->getAccessToken();
        if (!$accessToken) return null;

        $body = [
            'dimensions' => [['name' => 'eventName']],
            'metrics' => [['name' => 'eventCount']]
        ];

        try {
            $res = $this->client->post('https://analyticsdata.googleapis.com/v1beta/properties/' . $this->propertyId . ':runRealtimeReport', [
                'headers' => [
                    'Authorization' => 'Bearer ' . $accessToken,
                    'Content-Type' => 'application/json'
                ],
                'body' => json_encode($body),
                'timeout' => 15
            ]);

            $data = json_decode((string) $res->getBody(), true);

            // Sum only rows matching the page_view event
            if (!empty($data['rows'])) {
                $sum = 0;
                foreach ($data['rows'] as $row) {
                    $eventName = $row['dimensionValues'][0]['value'] ?? null;
                    $metricVal = $row['metricValues'][0]['value'] ?? null;
                    if ($eventName === 'page_view' && $metricVal !== null) {
                        $sum += (int) $metricVal;
                    }
                }
                // If we found a sum for page_view, return it, otherwise null
                return $sum > 0 ? $sum : null;
            }

            if (!empty($data['totals'][0]['values'][0])) {
                return (int) $data['totals'][0]['values'][0];
            }

            return null;
        } catch (\Exception $e) {
            return null;
        }
    }

    /**
     * Return a daily pageview time series between $from and $to as ['labels'=>[], 'values'=>[]]
     * @param Carbon|null $from
     * @param Carbon|null $to
     * @return array|null
     */
    public function getPageViewsSeries($from = null, $to = null)
    {
        if (!$this->hasConfig()) {
            return null;
        }

        $accessToken = $this->getAccessToken();
        if (!$accessToken) return null;

        // default to last 30 days
        if (!$from || !$to) {
            $to = Carbon::today();
            $from = Carbon::today()->subDays(29);
        }

        $startDate = $from->toDateString();
        $endDate = $to->toDateString();

        // Request eventCount per date and eventName, then filter to page_view
        $body = [
            'dateRanges' => [ ['startDate' => $startDate, 'endDate' => $endDate] ],
            'dimensions' => [ ['name' => 'date'], ['name' => 'eventName'] ],
            'metrics' => [ ['name' => 'eventCount'] ],
            'limit' => 10000
        ];

        try {
            $res = $this->client->post('https://analyticsdata.googleapis.com/v1beta/properties/' . $this->propertyId . ':runReport', [
                'headers' => [
                    'Authorization' => 'Bearer ' . $accessToken,
                    'Content-Type' => 'application/json'
                ],
                'body' => json_encode($body),
                'timeout' => 20
            ]);

            $data = json_decode((string) $res->getBody(), true);
            $series = [];
            // initialize series with zeros for each date
            $period = new \DatePeriod(new \DateTime($startDate), new \DateInterval('P1D'), (new \DateTime($endDate))->modify('+1 day'));
            foreach ($period as $dt) {
                $d = $dt->format('Y-m-d');
                $series[$d] = 0;
            }

            if (!empty($data['rows'])) {
                foreach ($data['rows'] as $row) {
                    $date = $row['dimensionValues'][0]['value'] ?? null;
                    $event = $row['dimensionValues'][1]['value'] ?? null;
                    $val = $row['metricValues'][0]['value'] ?? 0;
                    if ($date && $event === 'page_view') {
                        $series[$date] = ((int)$series[$date]) + (int)$val;
                    }
                }
            }

            return [ 'labels' => array_keys($series), 'values' => array_values($series) ];
        } catch (\Exception $e) {
            return null;
        }
    }

    /**
     * Return top pages by pageviews for a date range. Returns array of ['title' => ..., 'path' => ..., 'views' => int]
     * @param Carbon|null $from
     * @param Carbon|null $to
     * @param int $limit
     * @return array|null
     */
    public function getTopPages($from = null, $to = null, $limit = 10)
    {
        if (!$this->hasConfig()) {
            return null;
        }

        $accessToken = $this->getAccessToken();
        if (!$accessToken) return null;

        if (!$from || !$to) {
            $to = Carbon::today();
            $from = Carbon::today()->subDays(29);
        }

        $startDate = $from->toDateString();
        $endDate = $to->toDateString();

        // Request eventCount grouped by pageTitle/pagePath and include eventName
        $body = [
            'dateRanges' => [ ['startDate' => $startDate, 'endDate' => $endDate] ],
            'dimensions' => [ ['name' => 'pageTitle'], ['name' => 'pagePath'], ['name' => 'eventName'] ],
            'metrics' => [ ['name' => 'eventCount'] ],
            'orderBys' => [ ['metric' => ['metricName' => 'eventCount'], 'desc' => true] ],
            'limit' => (int) $limit
        ];

        try {
            $res = $this->client->post('https://analyticsdata.googleapis.com/v1beta/properties/' . $this->propertyId . ':runReport', [
                'headers' => [
                    'Authorization' => 'Bearer ' . $accessToken,
                    'Content-Type' => 'application/json'
                ],
                'body' => json_encode($body),
                'timeout' => 20
            ]);

            $data = json_decode((string) $res->getBody(), true);
            $out = [];
            if (!empty($data['rows'])) {
                foreach ($data['rows'] as $row) {
                    $title = $row['dimensionValues'][0]['value'] ?? '(no title)';
                    $path = $row['dimensionValues'][1]['value'] ?? null;
                    $event = $row['dimensionValues'][2]['value'] ?? null;
                    $views = isset($row['metricValues'][0]['value']) ? (int) $row['metricValues'][0]['value'] : 0;
                    // include only page_view events
                    if ($event === 'page_view') {
                        $out[] = ['title' => $title, 'path' => $path, 'views' => $views];
                    }
                }
            }
            return $out;
        } catch (\Exception $e) {
            return null;
        }
    }

    /**
     * Request realtime top pages using the Realtime API (GA4).
     * Returns array of ['title'=>..., 'path'=>..., 'views'=>int] or null on failure.
     */
    public function getRealtimeTopPages($limit = 50)
    {
        if (!$this->hasConfig()) {
            return null;
        }

        $accessToken = $this->getAccessToken();
        if (!$accessToken) return null;

        // Request eventCount broken down by pageTitle/pagePath
        $body = [
            'dimensions' => [ ['name' => 'pageTitle'], ['name' => 'pagePath'], ['name' => 'eventName'] ],
            'metrics' => [ ['name' => 'eventCount'] ],
            'limit' => (int) $limit
        ];

        try {
            $res = $this->client->post('https://analyticsdata.googleapis.com/v1beta/properties/' . $this->propertyId . ':runRealtimeReport', [
                'headers' => [
                    'Authorization' => 'Bearer ' . $accessToken,
                    'Content-Type' => 'application/json'
                ],
                'body' => json_encode($body),
                'timeout' => 10
            ]);

            $data = json_decode((string) $res->getBody(), true);
            $out = [];
            if (!empty($data['rows'])) {
                foreach ($data['rows'] as $row) {
                    $title = $row['dimensionValues'][0]['value'] ?? '(no title)';
                    $path = $row['dimensionValues'][1]['value'] ?? null;
                    $event = $row['dimensionValues'][2]['value'] ?? null;
                    $views = isset($row['metricValues'][0]['value']) ? (int) $row['metricValues'][0]['value'] : 0;
                    if ($event === 'page_view') {
                        $out[] = ['title' => $title, 'path' => $path, 'views' => $views];
                    }
                }
            }
            return $out;
        } catch (\Exception $e) {
            return null;
        }
    }

    /**
     * Detailed diagnostics helper for debugging connectivity and permissions.
     * Returns an array with either 'access_token' (masked) and 'api_result' or an 'error' message.
     * This method is safe to call from diagnostics routes since it does not expose private_key.
     */
    public function diagnose()
    {
        if (!$this->hasConfig()) {
            return ['error' => 'service account config missing or invalid'];
        }

        $accessToken = $this->getAccessToken();
        if (!$accessToken) {
            return ['error' => 'could not obtain access token'];
        }

        // Mask token when returning
        $masked = substr($accessToken, 0, 8) . '...' . substr($accessToken, -8);

        // Make a simple runReport request for a short date range to verify permissions
        $to = \Carbon\Carbon::today();
        $from = $to->subDays(1);
        $body = [
            'dateRanges' => [ ['startDate' => $from->toDateString(), 'endDate' => $to->toDateString()] ],
            'metrics' => [['name' => 'totalUsers']]
        ];

        try {
            $res = $this->client->post('https://analyticsdata.googleapis.com/v1beta/properties/' . $this->propertyId . ':runReport', [
                'headers' => [
                    'Authorization' => 'Bearer ' . $accessToken,
                    'Content-Type' => 'application/json'
                ],
                'body' => json_encode($body),
                'timeout' => 15
            ]);

            $status = $res->getStatusCode();
            $data = json_decode((string) $res->getBody(), true);
            return ['access_token' => $masked, 'status' => $status, 'api_result' => $data];
        } catch (\Exception $e) {
            return ['access_token' => $masked, 'error' => $e->getMessage()];
        }
    }
}
