<?php

namespace App\Http\Controllers;
use Carbon\Carbon;
use App\Models\News;
use App\Models\Page;
use App\Models\Plan;
use App\Models\Career;
use App\Models\Service;
use App\Models\Frontend;
use App\Models\User;
use App\Models\Deposit;
use App\Models\ReportDownload;
use App\Models\Order;
use App\Models\Language;
use App\Models\Portfolio;
use App\Models\Subscriber;
use Illuminate\Http\Request;
use App\Models\SupportTicket;
use App\Models\GeneralSetting;
use App\Models\SupportMessage;
use App\Models\ReportAndInside;
use App\Models\OfficeEventSpace;
use App\Models\AdminNotification;
use App\Models\OfficeEventSpaceSlide;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Cache;
use Illuminate\Support\Facades\Cookie;
use App\Services\GoogleAnalyticsService;

class SiteController extends Controller
{
    public function index(){
        $template = @$_GET['template'];
        if ($template) {
            $general = gs();
            $general->active_template = $template;
            Cache::put('GeneralSetting', $general);
        }

        $pageTitle = 'Home';
        $sections = Page::where('tempname',$this->activeTemplate)->where('slug','/')->first();
        // Totals for frontend counters (used by counter section)
        $totals = [
            'users' => User::count(),
            'services' => Service::count(),
            'portfolios' => Portfolio::count(),
            'downloads' => ReportDownload::count(),
            'deposits' => Deposit::successful()->sum('amount'),
            'orders' => Order::count(),
            'viewers' => null, // filled from Google Analytics when configured
        ];

        // If Google Analytics is enabled and configured, try to fetch total viewers
        try {
            $ga = new GoogleAnalyticsService();
            $gaTotal = $ga->getTotalUsers();
            if (is_numeric($gaTotal)) {
                $totals['viewers'] = (int) $gaTotal;
            }
        } catch (\Exception $e) {
            // ignore and fall back to default totals
        }
        return view($this->activeTemplate . 'home', compact('pageTitle','sections','totals'));
    }

    public function pages($slug)
    {
        $page = Page::where('tempname',$this->activeTemplate)->where('slug',$slug)->firstOrFail();

        $pageHeading = $page->sub_title;
        $pageCover = asset('assets/images/frontend/others/bg-2.jpg');

        $pageTitle = $page->name;
        $sections = $page->secs;
        return view($this->activeTemplate . 'pages', compact('pageTitle', 'pageHeading', 'pageCover','sections'));
    }


    public function contact()
    {
        $page = Page::where('tempname',$this->activeTemplate)->where('slug', 'contact')->firstOrFail();
        $pageHeading = $page->sub_title;
        // $pageCover = asset('assets/images/frontend/others/bg-2.jpg');
        $pageCover = getImage(getFilePath('pageCoverImage').'/'.$page->cover_image);
        $pageTitle = $page->name;
        return view($this->activeTemplate . 'contact',compact('pageTitle', 'pageHeading', 'pageCover'));
    }

    public function getLandingPage(){
        $pageTitle = "Prestige";
        return view('landing', compact('pageTitle'));
    }
    public function contactSubmit(Request $request)
    {
        $this->validate($request, [
            'name' => 'required',
            'email' => 'required',
            // 'subject' => 'required|string|max:255',
            'message' => 'required',
        ]);

        if(!verifyCaptcha()){
            $notify[] = ['error','Invalid captcha provided'];
            return back()->withNotify($notify);
        }

        $request->session()->regenerateToken();

        $random = getNumber();

        $setting = GeneralSetting::first();
        $content = Frontend::where('data_keys', 'contact_us.content')->orderBy('id','desc')->first();

        $subject = 'New Contact Request';

        $ticket = new SupportTicket();
        $ticket->user_id = auth()->id() ?? 0;
        $ticket->name = $request->name;
        $ticket->email = $request->email;
        $ticket->priority = 2;


        $ticket->ticket = $random;
        $ticket->subject = $subject;
        $ticket->last_reply = Carbon::now();
        $ticket->status = 0;
        $ticket->save();

        $adminNotification = new AdminNotification();
        $adminNotification->user_id = auth()->user() ? auth()->user()->id : 0;
        $adminNotification->title = 'A new support ticket has opened ';
        $adminNotification->click_url = urlPath('admin.ticket.view',$ticket->id);
        $adminNotification->save();

        $message = new SupportMessage();
        $message->support_ticket_id = $ticket->id;
        $message->message = $request->message;
        $message->save();

        // Mail Send

        // To User
        $receiverName = explode('@', $request->email)[0];
        $user = [
            'username'=> $request->email,
            'email'=> $request->email,
            'fullname'=> $receiverName,
        ];

        notify($user,'CONTACT',[
            'subject' => $subject,
            'message' => $request->body,
            'user_name' => $request->name,
            'user_email' => $request->email
        ],['email']);

        // To Admin
        $admin = [
            'username'=> $content->data_values->email_address,
            'email'=> $content->data_values->email_address,
            'fullname'=> $setting->site_name,
        ];
        notify($admin,'CONTACT_ADMIN',[
            'subject' => $subject,
            'message' => $request->body,
            'user_name' => $request->name,
            'user_email' => $content->data_values->email_address
        ],['email']);

        $notify[] = ['success', 'Ticket created successfully!'];

        // return to_route('ticket.view', [$ticket->ticket])->withNotify($notify);
        return back()->withNotify($notify);
    }

    public function policyPages($slug,$id)
    {
        $policy = Frontend::where('id',$id)->where('data_keys','policy_pages.element')->firstOrFail();
        $pageTitle = $policy->data_values->title;
        return view($this->activeTemplate.'policy',compact('policy','pageTitle'));
    }

    public function changeLanguage($lang = null)
    {
        $language = Language::where('code', $lang)->first();
        if (!$language) $lang = 'en';
        session()->put('lang', $lang);
        return back();
    }

    public function blog(){
        $sections = Page::where('tempname',$this->activeTemplate)->where('slug','blog')->firstOrFail();
        // $pageCover = asset('assets/images/frontend/others/bg-2.jpg');
        $pageCover = getImage(getFilePath('pageCoverImage').'/'.$sections->cover_image);

        $pageTitle = $sections->name;
        $pageHeading = $sections->sub_title;

        $featuredBlog = News::where('status', 1)->orderBy('created_at','desc')->first();
        $blogs = News::where('status', 1)
            ->when($featuredBlog, function($query) use ($featuredBlog) {
                $query->whereNot('id', $featuredBlog->id);
            })
            ->orderBy('created_at','desc')
            ->paginate(6);
        return view($this->activeTemplate.'blog.index',compact('sections','blogs', 'featuredBlog','pageTitle', 'pageHeading', 'pageCover'));
    }

    public function blogDetails($slug){
        $blog = News::where('status',1)->where('slug',$slug)->firstOrFail();

        // Optionally: prevent duplicate views by same session
        $blog->increment('view_count');
        $key = 'blog_viewed_' . $blog->id;
        if (!session()->has($key)) {
            // $blog->increment('view_count');
            session()->put($key, true);
        }

        $pageTitle = $blog->title;
        $pageImage = getImage(getFilePath('blog').'/'. $blog->featured_image);
        $pageDescription = \Str::limit(strip_tags($blog->description), 160);
        $related_articles = News::where('status',1)
            ->whereNot('id', $blog->id)->inRandomOrder()->limit(3)->get();
        return view($this->activeTemplate.'blog.details',compact('related_articles','blog','pageTitle', 'pageImage', 'pageDescription'));
    }

    public function services(){
        // $pageCover = asset('assets/images/frontend/others/bg-2.jpg');
        $sections = Page::where('tempname',$this->activeTemplate)->where('slug','services')->firstOrFail();
        $services = Service::where('status',1)->orderBy('created_at','desc')->paginate(getPaginate());

        $pageCover = getImage(getFilePath('pageCoverImage').'/'.$sections->cover_image);

        $pageTitle = $sections->name;
        $pageHeading = $sections->sub_title;

        return view($this->activeTemplate.'services.service',compact('sections','services','pageTitle', 'pageHeading', 'pageCover'));
    }

    public function serviceDetails($slug,$id){
        $service = Service::where('id',$id)->firstOrFail();
        $pageTitle = $service->title;
        $latests = Service::inRandomOrder()->limit(5)->get();
        return view($this->activeTemplate.'services.service_details',compact('latests','service','pageTitle'));
    }

    public function portfolio(){

        $page = Page::where('tempname',$this->activeTemplate)->where('slug','portfolio')->firstOrFail();
        $pageTitle = $page->name;
        $pageHeading = $page->sub_title;
        $pageCover = getImage(getFilePath('pageCoverImage').'/'.$page->cover_image);

        $sections = Page::where('tempname',$this->activeTemplate)->where('slug','portfolio')->firstOrFail();
        $portfolio = getContent('portfolio.content',true);
        $portfolios = Portfolio::where('status',1)->orderBy('created_at','desc')->paginate(12);
        return view($this->activeTemplate.'portfolio.portfolio',compact('sections','portfolio', 'portfolios','pageTitle', 'pageHeading', 'pageCover'));
    }

    public function portfolioDetails($slug,$id){
        $portfolio = Portfolio::where('id',$id)->firstOrFail();
        $pageTitle = $portfolio->title;
        $latests = Portfolio::inRandomOrder()->limit(5)->get();
        return view($this->activeTemplate.'portfolio.portfolio_details',compact('latests','portfolio','pageTitle'));
    }

    // public function blogDetails($slug,$id){
    //     $blog = Frontend::where('id',$id)->where('data_keys','blog.element')->firstOrFail();
    //     $pageTitle = $blog->data_values->title;
    //     $latests = Frontend::where('data_keys','blog.element')->orderBy('id','desc')->limit(5)->get();
    //     return view($this->activeTemplate.'blog_details',compact('latests','blog','pageTitle'));
    // }

    // plan
    public function plan(){
        $pageTitle = 'Plans';
        $sections = Page::where('tempname',$this->activeTemplate)->where('slug','plan')->firstOrFail();
        $plans = Plan::where('status',1)->orderBy('created_at','desc')->paginate(getPaginate());
        return view($this->activeTemplate.'plan.plan',compact('pageTitle','plans','sections'));
    }

    public function subscribe(Request $request){

        $request->validate([
            'email'=>'required|unique:subscribers',
        ]);

        $subscribe=new Subscriber();
        $subscribe->email=$request->email;
        $subscribe->save();

        $notify[] = ['success','You have successfully subscribed to the Newsletter'];
        return back()->withNotify($notify);

    }


    public function cookieAccept(){
        $general = gs();
        Cookie::queue('gdpr_cookie',$general->site_name , 43200);
        return back();
    }

    public function cookiePolicy(){
        $pageTitle = 'Cookie Policy';
        $cookie = Frontend::where('data_keys','cookie.data')->first();
        return view($this->activeTemplate.'cookie',compact('pageTitle','cookie'));
    }

    public function placeholderImage($size = null){
        $imgWidth = explode('x',$size)[0];
        $imgHeight = explode('x',$size)[1];
        $text = $imgWidth . '×' . $imgHeight;
        $fontFile = realpath('assets/font') . DIRECTORY_SEPARATOR . 'RobotoMono-Regular.ttf';
        $fontSize = round(($imgWidth - 50) / 8);
        if ($fontSize <= 9) {
            $fontSize = 9;
        }
        if($imgHeight < 100 && $fontSize > 30){
            $fontSize = 30;
        }

        $image     = imagecreatetruecolor($imgWidth, $imgHeight);
        $colorFill = imagecolorallocate($image, 255, 255, 255);
        $bgFill    = imagecolorallocate($image, 28, 35, 47);
        imagefill($image, 0, 0, $bgFill);
        $textBox = imagettfbbox($fontSize, 0, $fontFile, $text);
        $textWidth  = abs($textBox[4] - $textBox[0]);
        $textHeight = abs($textBox[5] - $textBox[1]);
        $textX      = ($imgWidth - $textWidth) / 2;
        $textY      = ($imgHeight + $textHeight) / 2;
        header('Content-Type: image/jpeg');
        imagettftext($image, $fontSize, 0, $textX, $textY, $colorFill, $fontFile, $text);
        imagejpeg($image);
        imagedestroy($image);
    }

    // About
    public function about(){
        $page = Page::where('tempname',$this->activeTemplate)->where('slug', 'about')->firstOrFail();
        $pageHeading = $page->sub_title;
        // $pageCover = asset('assets/images/frontend/others/bg-2.jpg');
        $pageCover = getImage(getFilePath('pageCoverImage').'/'.$page->cover_image);

        $pageTitle = $page->name;

        $sections = Page::where('tempname',$this->activeTemplate)->where('slug','about')->firstOrFail();
        $aboutElements = Frontend::where('data_keys','about.element')->orderBy('id','desc')->get();

        return view($this->activeTemplate.'about',compact('sections', 'aboutElements','pageTitle', 'pageHeading', 'pageCover'));
    }

    // Report and Inside report-and-inside
    public function reportAndInside(){
        $sections = Page::where('tempname',$this->activeTemplate)->where('slug','reports-and-insights')->firstOrFail();
        $pageTitle = $sections->name;
        $pageHeading = $sections->sub_title;
        $pageCover = getImage(getFilePath('pageCoverImage').'/'.$sections->cover_image);

        $reports = ReportAndInside::with('category')->where('status', 1)->orderBy('created_at','desc')->paginate(12);
        return view($this->activeTemplate.'report-insides.index',compact('sections', 'reports','pageTitle', 'pageHeading', 'pageCover'));
    }

    public function reportAndInsideDetails($slug){
        $report = ReportAndInside::with('category')->where('slug',$slug)->firstOrFail();

        // Optionally: prevent duplicate views by same session
        $report->increment('view_count');
        $key = 'report_viewed_' . $report->id;
        if (!session()->has($key)) {
            // $report->increment('view_count');
            session()->put($key, true);
        }


        $pageTitle = $report->title;
        $latests = ReportAndInside::with('category')->inRandomOrder()->limit(5)->get();
        return view($this->activeTemplate.'report-insides.details',compact('latests','report','pageTitle'));
    }

    // Careers
    public function careerCenter(){

        $is_internal = request()->get('prestige');

        $sections = Page::where('tempname',$this->activeTemplate)->where('slug','career-center')->firstOrFail();
        $pageTitle = $sections->name;
        $pageHeading = $sections->sub_title;
        $pageCover = getImage(getFilePath('pageCoverImage').'/'.$sections->cover_image);

        $rows = Career::where('status', 1)
            ->when($is_internal, function($query) use ($is_internal) {
                $query->where('is_internal', 1);
            })
            ->orderBy('created_at','desc')
            ->paginate(10);
        return view($this->activeTemplate.'careers.index',compact('sections', 'rows','pageTitle', 'pageHeading', 'pageCover'));
    }

    public function careerCenterDetails($uuid){
        $career = Career::where('uuid',$uuid)->firstOrFail();
        $pageTitle = $career->title;
        return view($this->activeTemplate.'careers.details',compact('career','pageTitle'));
    }

    // Office and Event Space
    public function officeEventSpace(){
        $sections = Page::where('tempname',$this->activeTemplate)->where('slug','office-and-event-space')->firstOrFail();
        $pageTitle = $sections->name;
        $pageHeading = $sections->sub_title;
        $pageCover = getImage(getFilePath('pageCoverImage').'/'.$sections->cover_image);
        $officeEventSpace = getContent('office_and_event_space.content',true);
        $officeEventSpaceElements = getContent('office_and_event_space.element',false);

        $offices = OfficeEventSpace::where('type', OfficeEventSpace::TYPE_OFFICE)->where('status',1)->orderBy('created_at', 'DESC')->get();
        $events = OfficeEventSpace::where('type', OfficeEventSpace::TYPE_EVENT)->where('status',1)->orderBy('created_at', 'DESC')->get();
        $slides = OfficeEventSpaceSlide::where('status', 1)->orderBy('created_at')->get();

        return view($this->activeTemplate.'office_and_event_space.index',compact('sections', 'officeEventSpace', 'officeEventSpaceElements', 'offices', 'events', 'slides','pageTitle', 'pageHeading', 'pageCover'));
    }
}
