<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\Mail;
use App\Models\ReportDownload;
use App\Models\ReportAndInside;

class ReportDownloadController extends Controller
{
    public function store(Request $request)
    {
        \Log::info('Download request received:', $request->all());

        // Validate input
        $validated = $request->validate([
            'report_id'   => 'required|integer|exists:report_and_insides,id',
            'first_name'  => 'required|string|max:255',
            'last_name'   => 'required|string|max:255',
            'email'       => 'required|email|max:255',
        ]);

        \Log::info('Validation passed:', $validated);

        // Get report file for download and snapshot its fields
        $report = ReportAndInside::findOrFail($request->report_id);

        $snapshot = [
            'download_report_file' => $report->download_report_file,
            'download_report_external_url' => $report->download_report_external_url,
        ];

        // Save record (include snapshot fields)
        $download = ReportDownload::create(array_merge($validated, $snapshot));

        \Log::info('Download record created:', ['id' => $download->id]);

        // Prefer external URL when provided; otherwise use stored file snapshot
        if (!empty($download->download_report_external_url)) {
            $downloadUrl = $download->download_report_external_url;
        } elseif (!empty($download->download_report_file)) {
            $downloadUrl = asset('assets/images/frontend/reportInsideImage/' . $download->download_report_file);
        } else {
            $downloadUrl = null;
        }

        // Send emails
        try {
            // Email to user
            $this->sendUserEmail($request, $report, $downloadUrl);

            // Email to admin
            $this->sendAdminNotification($request, $report);
        } catch (\Exception $e) {
            // Log error but don't block download
            \Log::error('Email sending failed: ' . $e->getMessage());
        }

        return response()->json([
            'success' => true,
            'download_url' => $downloadUrl,
            'message' => 'Thank you! Check your email for the download link.'
        ]);
    }

    private function sendUserEmail($request, $report, $downloadUrl)
    {
        $data = [
            'name' => $request->first_name . ' ' . $request->last_name,
            'first_name' => $request->first_name,
            'report_title' => $report->title,
            'download_url' => $downloadUrl
        ];

        Mail::send('emails.report_download_user', $data, function($message) use ($request, $report) {
            $message->to($request->email, $request->first_name . ' ' . $request->last_name)
                    ->subject('Your Report: ' . $report->title);
        });
    }

    private function sendAdminNotification($request, $report)
    {
        $adminEmail = config('mail.from.address') ?? 'info@prestigealliance.co';

        $data = [
            'first_name' => $request->first_name,
            'last_name' => $request->last_name,
            'email' => $request->email,
            'report_title' => $report->title,
            'downloaded_at' => now()->format('F d, Y H:i:s')
        ];

        Mail::send('emails.report_download_admin', $data, function($message) use ($adminEmail, $report) {
            $message->to($adminEmail)
                    ->subject('New Report Download: ' . $report->title);
        });
    }
}
