<?php

namespace App\Http\Controllers\Admin;

use Carbon\Carbon;
use App\Models\Plan;
use App\Models\User;
use App\Models\Order;
use App\Models\Deposit;
use App\Models\Service;
use App\Lib\CurlRequest;
use App\Models\Portfolio;
use App\Models\UserLogin;
use App\Models\Transaction;
use App\Models\Subscription;
use App\Services\GoogleAnalyticsService;
use Illuminate\Http\Request;
use App\Models\SupportTicket;
use App\Rules\FileTypeValidate;
use App\Models\AdminNotification;
use App\Models\ReportDownload;
use App\Models\PageView;
use Illuminate\Http\JsonResponse;
use App\Http\Controllers\Controller;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Cache;

class AdminController extends Controller
{
    public function upload(Request $request): JsonResponse
    {
        try {
            if ($request->hasFile('upload')) {
                $originName = $request->file('upload')->getClientOriginalName();
                $fileName = pathinfo($originName, PATHINFO_FILENAME);
                $extension = $request->file('upload')->getClientOriginalExtension();
                $fileName = $fileName . '_' . time() . '.' . $extension;

                $url = fileUploader($request->upload, getFilePath('uploads'));
                // $url = getImage(getFilePath('uploads').'/'.$url);
                $baseUrl = '/' . getFilePath('uploads').'/'.$url;

                // return response()->json(['fileName' => $fileName, 'uploaded'=> 1, 'url' => $url]);
                return response()->json([
                    'url' => $baseUrl
                ]);
            }
        } catch (\Exception $exp) {
            $notify[] = ['error', 'Couldn\'t upload your image'];
            return back()->withNotify($notify);
        }
    }

    public function dashboard(Request $request)
    {

        $pageTitle = 'Dashboard';

        // User Info
        $widget['total_users']             = User::count();
        $widget['verified_users']          = User::where('status', 1)->where('ev',1)->where('sv',1)->count();
        $widget['email_unverified_users']  = User::emailUnverified()->count();
        $widget['mobile_unverified_users'] = User::mobileUnverified()->count();


        $deposit['total_deposit_amount']        = Deposit::successful()->sum('amount');
        $deposit['total_deposit_pending']       = Deposit::pending()->count();
        $deposit['total_deposit_rejected']      = Deposit::rejected()->count();
        $deposit['total_deposit_charge']        = Deposit::successful()->sum('charge');

        // Date range filter (optional)
        $from = null;
        $to = null;
        $range = $request->get('range');
        if ($range) {
            switch ($range) {
                case 'today':
                    $from = Carbon::today()->startOfDay();
                    $to = Carbon::today()->endOfDay();
                    break;
                case 'week':
                    $from = Carbon::now()->startOfWeek();
                    $to = Carbon::now()->endOfWeek();
                    break;
                case 'month':
                    $from = Carbon::now()->startOfMonth();
                    $to = Carbon::now()->endOfMonth();
                    break;
                case 'year':
                    $from = Carbon::now()->startOfYear();
                    $to = Carbon::now()->endOfYear();
                    break;
                case 'custom':
                    if ($request->filled('from') && $request->filled('to')) {
                        try {
                            $from = Carbon::parse($request->get('from'))->startOfDay();
                            $to = Carbon::parse($request->get('to'))->endOfDay();
                        } catch (\Exception $e) {
                            $from = null; $to = null;
                        }
                    }
                    break;
            }
        }

        // Log incoming range/from/to for debugging date filter issues
        Log::info('Admin dashboard range filter', ['range' => $range, 'from' => $from?->toDateString(), 'to' => $to?->toDateString()]);

        $totalPlan = Plan::count();
        $totalService = $from ? Service::whereBetween('created_at', [$from, $to])->count() : Service::count();
        $totalPortfolio = $from ? Portfolio::whereBetween('created_at', [$from, $to])->count() : Portfolio::count();
        $totalDownloads = $from ? ReportDownload::whereBetween('created_at', [$from, $to])->count() : ReportDownload::count();

        // Determine GA date range to use for queries.
        // If no explicit range selected (All Time), use a wide start date so totals are cumulative
        // instead of the service's default rolling 30-day window. This prevents the "decreasing"
        // total caused by a sliding 30-day window when the UI label says "All Time".
        $gaFrom = $from;
        $gaTo = $to;
        if (empty($range)) {
            // Use a safe early date for 'All Time'. Adjust if you prefer a different epoch.
            $gaFrom = Carbon::create(2005, 1, 1)->startOfDay();
            $gaTo = Carbon::today()->endOfDay();
        }

        // Attempt to get page views from Google Analytics (GA4). If GA isn't configured
        // or the API call fails, fall back to the DB user count (legacy fallback).
        $gaPageViews = null;
        try {
            $gaService = new GoogleAnalyticsService();
            $cacheKeyGaTotal = 'admin:ga:pageviews:' . ($range ?: 'all') . ':' . ($from?->toDateString() ?: '') . ':' . ($to?->toDateString() ?: '');
            // Use shorter cache for 'today' to keep numbers fresh, otherwise 5 minutes.
            $gaCacheTtl = ($range === 'today') ? 30 : 300;
            $gaPageViews = Cache::remember($cacheKeyGaTotal, $gaCacheTtl, function () use ($gaService, $gaFrom, $gaTo) {
                try {
                    return $gaService->getPageViews($gaFrom, $gaTo);
                } catch (\Exception $e) {
                    return null;
                }
            });
        } catch (\Exception $e) {
            $gaPageViews = null;
        }

        // Track which source provided the pageViews value for logging/debugging
        $pageViewsSource = null;

        $cacheBase = 'admin:pageviews:' . ($range ?: 'all') . ':' . ($from?->toDateString() ?: '') . ':' . ($to?->toDateString() ?: '');

        if ($gaPageViews !== null) {
            // If GA is available, still use the local cumulative DB total (excluding admin)
            // for the numeric dashboard card. This avoids GA counting historical admin hits.
            if (class_exists('\App\\Models\\PageView')) {
                $pageViews = Cache::remember($cacheBase . ':db_total', 60, function () {
                    return \App\Models\PageView::where('path', 'not like', 'admin%')
                        ->where('path', 'not like', '/admin%')
                        ->sum('views');
                });
                $pageViewsSource = 'db_cumulative_excluding_admin';
            } else {
                $pageViews = (int) $gaPageViews;
                $pageViewsSource = 'ga';
            }
        } else {
            // GA unavailable: try realtime, then DB fallbacks per range
            try {
                // Try realtime for 'today' or all-time default
                $pageViews = 0;
                if (empty($range) || $range === 'today') {
                    $realtime = null;
                    try {
                        $realtime = $gaService->getRealtimePageViews();
                    } catch (\Exception $inner) {
                        $realtime = null;
                    }

                    if ($realtime !== null) {
                        $pageViews = (int) $realtime;
                        $pageViewsSource = 'realtime';
                    } else {
                        // Use DB for today/all-time
                        if ($from) {
                            // range provided (today)
                            if (\Schema::hasTable('page_view_daily')) {
                                $pageViews = Cache::remember($cacheBase . ':daily_range', 60, function () use ($from, $to) {
                                    return \DB::table('page_view_daily')
                                        ->where('path', 'not like', 'admin%')
                                        ->where('path', 'not like', '/admin%')
                                        ->whereBetween('view_date', [$from->toDateString(), $to->toDateString()])
                                        ->sum('views');
                                });
                                $pageViewsSource = 'db_daily_range';
                            } else {
                                $pageViews = Cache::remember($cacheBase . ':logs_range', 60, function () use ($from, $to) {
                                    return \App\Models\PageViewLog::where('path', 'not like', 'admin%')
                                        ->where('path', 'not like', '/admin%')
                                        ->whereBetween('created_at', [$from, $to])
                                        ->count();
                                });
                                $pageViewsSource = 'db_logs_range';
                            }
                        } else {
                            // No explicit range: use cumulative PageView if present
                            if (class_exists('\App\\Models\\PageView')) {
                                $pageViews = Cache::remember($cacheBase . ':db_total', 60, function () {
                                    return \App\Models\PageView::where('path', 'not like', 'admin%')
                                        ->where('path', 'not like', '/admin%')
                                        ->sum('views');
                                });
                                $pageViewsSource = 'db_cumulative';
                            } else {
                                if (\Schema::hasTable('page_view_daily')) {
                                    $pageViews = \DB::table('page_view_daily')->sum('views');
                                    $pageViewsSource = 'db_daily_alltime';
                                } else {
                                    $pageViews = \App\Models\PageViewLog::count();
                                    $pageViewsSource = 'db_logs_alltime';
                                }
                            }
                        }
                    }
                } else {
                    // Explicit ranges (week/month/year/custom)
                    if (\Schema::hasTable('page_view_daily')) {
                        $pageViews = Cache::remember($cacheBase . ':daily_range', 60, function () use ($from, $to) {
                            return \DB::table('page_view_daily')
                                ->where('path', 'not like', 'admin%')
                                ->where('path', 'not like', '/admin%')
                                ->whereBetween('view_date', [$from->toDateString(), $to->toDateString()])
                                ->sum('views');
                        });
                        $pageViewsSource = 'db_daily_range';
                    } else {
                        $pageViews = Cache::remember($cacheBase . ':logs_range', 60, function () use ($from, $to) {
                            return \App\Models\PageViewLog::where('path', 'not like', 'admin%')
                                ->where('path', 'not like', '/admin%')
                                ->whereBetween('created_at', [$from, $to])
                                ->count();
                        });
                        $pageViewsSource = 'db_logs_range';
                    }
                }
            } catch (\Exception $e) {
                $pageViews = 0;
                $pageViewsSource = 'error';
            }
        }

        // Log which source was used and the computed pageViews value for debugging
        try {
            $db_including_admin = null;
            $db_excluding_admin = null;
            if (class_exists('\App\\Models\\PageView')) {
                $db_including_admin = \App\Models\PageView::sum('views');
                $db_excluding_admin = \App\Models\PageView::where('path', 'not like', 'admin%')
                    ->where('path', 'not like', '/admin%')
                    ->sum('views');
            }

            Log::info('Admin dashboard pageviews source', [
                'range' => $range,
                'from' => $from?->toDateString(),
                'to' => $to?->toDateString(),
                'pageViews' => $pageViews ?? null,
                'source' => $pageViewsSource,
                'ga_total' => $gaPageViews ?? null,
                'db_including_admin' => $db_including_admin,
                'db_excluding_admin' => $db_excluding_admin,
                'ga_range_total' => $gaRangePageViews ?? null,
            ]);
        } catch (\Throwable $ex) {
            // ignore logging failures
        }

        // Keep backwards compatibility: some views expect $totalUsers variable.
        $totalUsers = $pageViews ?? ($widget['total_users'] ?? User::count());
        $orders['total_order'] = Order::count();
        $orders['total_approved'] = Order::where('status',1)->count();
        $orders['total_pending'] = Order::where('status',0)->count();

        // subscriptions report graph
        $subscriptions = Subscription::where('created_at','>=',Carbon::now()->subDays(30))
        ->selectRaw("COUNT(plan_id) as total_subscriptions, DAY(created_at) as per_day" )
        ->orderBy('created_at')
        ->whereMonth('created_at', date('m'))
        ->groupByRaw("DAY(created_at)")
        ->pluck('total_subscriptions', 'per_day');

        $subscriptionsReport['labels']= $subscriptions->keys();
        $subscriptionsReport['values']= $subscriptions->values();
        // end subscriptions report graph

        // UserLogin Report Graph
        $userLoginsReport = UserLogin::selectRaw("COUNT(*) as created_at_count, DATE(created_at) as date_name")->orderBy('created_at', 'desc')
                    ->groupByRaw("DATE(created_at)")->limit(10)
                    ->pluck('created_at_count', 'date_name');
        $userLogins['labels'] = $userLoginsReport->keys();
        $userLogins['values'] = $userLoginsReport->values();

        // User Signups report (by date)
        if ($from) {
            $signupsQuery = User::whereBetween('created_at', [$from, $to]);
        } else {
            $signupsQuery = User::where('created_at', '>=', Carbon::now()->subDays(30));
        }

        $userSignupsReport = $signupsQuery->selectRaw("COUNT(*) as total, DATE(created_at) as date_name")
            ->groupByRaw('DATE(created_at)')
            ->orderBy('date_name')
            ->pluck('total', 'date_name');

        $userSignups['labels'] = $userSignupsReport->keys();
        $userSignups['values'] = $userSignupsReport->values();

        // UserLogin Report Graph
        // Eager-load first message to display a preview without extra queries
        $newTickets = SupportTicket::with(['user', 'supportMessageOne'])->orderBy('created_at', 'desc')->whereStatus(0)->limit(5)->get();

        $selectedRange = $range;
        $selectedFrom = $from ? $from->toDateString() : null;
        $selectedTo = $to ? $to->toDateString() : null;

            // --- Pageviews series for chart (labels / values)
        $pageViewsSeriesLabels = [];
        $pageViewsSeriesValues = [];
        try {
            if (!isset($gaService)) {
                $gaService = new GoogleAnalyticsService();
            }
            $series = $gaService->getPageViewsSeries($gaFrom, $gaTo);
            if (is_array($series)) {
                $pageViewsSeriesLabels = $series['labels'] ?? [];
                $pageViewsSeriesValues = $series['values'] ?? [];
            }
        } catch (\Exception $e) {
            $pageViewsSeriesLabels = [];
            $pageViewsSeriesValues = [];
        }

        // --- Active users (realtime)
        $activeUsers = null;
        try {
            if (!isset($gaService)) {
                $gaService = new GoogleAnalyticsService();
            }
            $activeUsers = $gaService->getRealtimeUsers();
        } catch (\Exception $e) {
            $activeUsers = null;
        }

            // --- Top pages (title, path, views)
            $topPages = [];
            try {
                $topPagesLimit = (int) env('GA_TOP_PAGES_LIMIT', 1000);
                if (!isset($gaService)) {
                    $gaService = new GoogleAnalyticsService();
                }

                // Helper: attach last_viewed_at from local PageView when available
                $attachLastViewed = function(array $row) {
                    if (!empty($row['path'])) {
                        $pv = PageView::where('path', $row['path'])->first();
                        if ($pv) {
                            $last = $pv->last_viewed_at ?? null;
                            $row['last_viewed_at'] = (is_string($last) || $last === null) ? $last : $last->toDateTimeString();
                        }
                    }
                    return $row;
                };

                if ($from) {
                    // Range requested: prefer GA range data
                    $rawTop = $gaService->getTopPages($gaFrom, $gaTo, $topPagesLimit) ?? [];
                    $gaRangePageViews = $gaService->getPageViews($gaFrom, $gaTo) ?? null;

                    if (!empty($rawTop)) {
                        foreach ($rawTop as $r) {
                            $gPath = isset($r['path']) ? ltrim($r['path'], '/') : '';
                            if ($gPath !== '' && str_starts_with($gPath, 'admin')) continue;
                            $row = $r;
                            $row['source'] = 'ga';
                            $row['last_viewed_at'] = null;
                            $topPages[] = $attachLastViewed($row);
                        }
                    } else {
                        // Try realtime, then daily aggregates / logs fallback
                        $realtimeTop = null;
                        try {
                            $realtimeTop = $gaService->getRealtimeTopPages(200);
                        } catch (\Exception $e) {
                            $realtimeTop = null;
                        }

                        if (!empty($realtimeTop)) {
                            foreach ($realtimeTop as $r) {
                                $gPath = isset($r['path']) ? ltrim($r['path'], '/') : '';
                                if ($gPath !== '' && str_starts_with($gPath, 'admin')) continue;
                                $row = $r;
                                $row['source'] = 'ga_realtime';
                                $row['last_viewed_at'] = null;
                                $topPages[] = $attachLastViewed($row);
                            }
                        } else {
                            if (\Schema::hasTable('page_view_daily')) {
                                $rawTop = \DB::table('page_view_daily')
                                    ->selectRaw('path, COALESCE(MAX(title), path) as title, SUM(views) as views, MAX(view_date) as last_viewed_at')
                                    ->where('path', 'not like', 'admin%')
                                    ->whereBetween('view_date', [$from->toDateString(), $to->toDateString()])
                                    ->groupBy('path')
                                    ->orderByDesc('views')
                                    ->limit($topPagesLimit)
                                    ->get()
                                    ->toArray();

                                foreach ($rawTop as $r) {
                                    $topPages[] = [
                                        'title' => $r->title ?? $r->path,
                                        'path' => $r->path ?? null,
                                        'views' => (int) ($r->views ?? 0),
                                        'source' => 'db',
                                        'last_viewed_at' => $r->last_viewed_at ?? null,
                                    ];
                                }
                            } else {
                                $rawTop = \App\Models\PageViewLog::selectRaw('path, COALESCE(MAX(title), path) as title, COUNT(*) as views, MAX(created_at) as last_viewed_at')
                                    ->where('path', 'not like', 'admin%')
                                    ->whereBetween('created_at', [$from, $to])
                                    ->groupBy('path')
                                    ->orderByDesc('views')
                                    ->limit($topPagesLimit)
                                    ->get()
                                    ->toArray();

                                foreach ($rawTop as $r) {
                                    $topPages[] = [
                                        'title' => $r['title'] ?? $r['path'],
                                        'path' => $r['path'] ?? null,
                                        'views' => (int) ($r['views'] ?? 0),
                                        'source' => 'db',
                                        'last_viewed_at' => $r['last_viewed_at'] ?? null,
                                    ];
                                }
                            }
                        }
                    }
                } else {
                    // No explicit range: use GA top pages for wide range
                    $rawTop = $gaService->getTopPages($gaFrom, $gaTo, $topPagesLimit) ?? [];
                    foreach ($rawTop as $r) {
                        $gPath = isset($r['path']) ? ltrim($r['path'], '/') : '';
                        if ($gPath !== '' && str_starts_with($gPath, 'admin')) continue;
                        $row = $r;
                        $row['source'] = 'ga';
                        $row['last_viewed_at'] = null;
                        $topPages[] = $attachLastViewed($row);
                    }
                }
            } catch (\Exception $e) {
                $topPages = [];
            }
                try {
                    if ($from) {
                        // don't merge cumulative PageView entries for explicit ranges
                        $dbPages = [];
                    } else {
                    if ($from) {
                        $dbQuery = PageView::where('path', 'not like', 'admin%')
                            ->whereBetween('last_viewed_at', [$from, $to])->orderBy('views', 'desc');
                    } else {
                        $dbQuery = PageView::where('path', 'not like', 'admin%')->orderBy('views', 'desc');
                    }

                    $dbPages = $dbQuery->get()
                        ->map(function($r){
                            $last = $r->last_viewed_at ?? null;
                            return [
                                'title' => $r->title ?? $r->path,
                                'path' => $r->path,
                                'views' => (int) $r->views,
                                'source' => 'db',
                                'last_viewed_at' => (is_string($last) || $last === null) ? $last : $last->toDateTimeString()
                            ];
                        })->toArray();
                }

                // Aggregate rows by path so duplicates (GA per-country rows or
                // DB entries) combine into a single row. We'll sum views,
                // merge source into 'mixed' when differing, and keep the most
                // recent last_viewed_at when available.
                $map = [];

                $addOrAggregate = function($p) use (&$map) {
                    $key = $p['path'] ?? ($p['title'] ?? null);
                    if ($key === null) return;

                    if (!isset($map[$key])) {
                        // normalize numeric views and initialize per-source counters
                        $p['views'] = (int) ($p['views'] ?? 0);
                        $p['ga_views'] = ($p['source'] ?? null) === 'ga' ? (int)$p['views'] : 0;
                        $p['db_views'] = ($p['source'] ?? null) === 'db' ? (int)$p['views'] : 0;
                        $map[$key] = $p;
                        return;
                    }

                    // sum views
                    $map[$key]['views'] = (int) ($map[$key]['views'] ?? 0) + (int) ($p['views'] ?? 0);

                    // maintain per-source counts
                    $map[$key]['ga_views'] = (int) ($map[$key]['ga_views'] ?? 0) + ((($p['source'] ?? null) === 'ga') ? (int)$p['views'] : 0);
                    $map[$key]['db_views'] = (int) ($map[$key]['db_views'] ?? 0) + ((($p['source'] ?? null) === 'db') ? (int)$p['views'] : 0);

                    // merge source
                    $existingSource = $map[$key]['source'] ?? null;
                    $newSource = $p['source'] ?? null;
                    if ($existingSource && $newSource && $existingSource !== $newSource) {
                        $map[$key]['source'] = 'mixed';
                    } elseif (empty($existingSource) && $newSource) {
                        $map[$key]['source'] = $newSource;
                    }

                    // preserve title if missing
                    if (empty($map[$key]['title']) && !empty($p['title'])) {
                        $map[$key]['title'] = $p['title'];
                    }

                    // keep most recent last_viewed_at
                    $a = $map[$key]['last_viewed_at'] ?? null;
                    $b = $p['last_viewed_at'] ?? null;
                    if ($a && $b) {
                        $map[$key]['last_viewed_at'] = (strtotime($b) > strtotime($a)) ? $b : $a;
                    } elseif ($b) {
                        $map[$key]['last_viewed_at'] = $b;
                    }
                };

                foreach ($topPages as $p) {
                    $addOrAggregate($p);
                }

                foreach ($dbPages as $p) {
                    $addOrAggregate($p);
                }

                // Preserve ordering by views desc (no slice - show all aggregated pages)
                $combined = array_values($map);
                usort($combined, function($a, $b){
                    return ((int)($b['views'] ?? 0)) <=> ((int)($a['views'] ?? 0));
                });
                $topPages = $combined;
            } catch (\Exception $e) {
                // ignore DB merge errors
            }

            // If no top pages were found for the explicit range (GA + range DB tables returned empty),
            // fall back to showing the cumulative `page_views` table so the UI isn't empty.
            // This provides a useful fallback especially for 'Today' when there may be no hits yet.
            try {
                if (empty($topPages) || count($topPages) === 0) {
                    if (class_exists('\\App\\Models\\PageView')) {
                        $fallbackLimit = intval(env('GA_TOP_PAGES_LIMIT', 1000));
                        $dbFallback = \App\Models\PageView::where('path', 'not like', 'admin%')
                            ->orderByDesc('views')
                            ->limit($fallbackLimit)
                            ->get()
                            ->map(function($r){
                                $last = $r->last_viewed_at ?? null;
                                return [
                                    'title' => $r->title ?? $r->path,
                                    'path' => $r->path,
                                    'views' => (int) $r->views,
                                    'source' => 'db_cumulative_fallback',
                                    'last_viewed_at' => (is_string($last) || $last === null) ? $last : $last->toDateTimeString()
                                ];
                            })->toArray();

                        if (!empty($dbFallback)) {
                            $topPages = $dbFallback;
                        }
                    }
                }
            } catch (\Exception $e) {
                // ignore fallback failures
            }

        // --- User pie chart (verified vs unverified)
        $userPieLabels = ['Verified', 'Unverified'];
        $totalUsersCount = $widget['total_users'] ?? User::count();
        $verifiedCount = $widget['verified_users'] ?? User::where('status',1)->where('ev',1)->where('sv',1)->count();
        $unverifiedCount = max(0, $totalUsersCount - $verifiedCount);
        $userPieValues = [$verifiedCount, $unverifiedCount];

        // Debug: log which data sources were used to generate the dashboard
        try {
            $tpSources = [];
            foreach ($topPages as $t) {
                if (isset($t['source'])) $tpSources[] = $t['source'];
            }
            $tpSources = array_values(array_unique($tpSources));
            Log::info('Admin dashboard data sources', [
                'range' => $range,
                'from' => $from?->toDateString(),
                'to' => $to?->toDateString(),
                'pageViews_count' => $pageViews ?? null,
                'top_pages_sources' => $tpSources,
                'top_pages_count' => count($topPages)
            ]);
        } catch (\Exception $e) {
            Log::error('Admin dashboard data sources log failed: ' . $e->getMessage());
        }

        return view('admin.dashboard', compact(
            'totalPlan', 'totalService', 'totalPortfolio', 'totalDownloads', 'totalUsers', 'pageViews',
            'subscriptionsReport', 'pageTitle', 'widget', 'deposit', 'userLogins', 'newTickets', 'orders',
            'selectedRange', 'selectedFrom', 'selectedTo', 'userSignups',
            'pageViewsSeriesLabels', 'pageViewsSeriesValues', 'userPieLabels', 'userPieValues', 'activeUsers', 'topPages'
        ));
    }

    /**
     * Import top pages from Google Analytics into the local `page_views` table.
     * This helps ensure pages reported by GA appear in the Top Pages list.
     */
    public function importGATopPages(Request $request)
    {
        $notify = [];
        try {
            $gaService = new GoogleAnalyticsService();
            $limit = (int) env('GA_TOP_PAGES_LIMIT', 1000);
            $gaRows = $gaService->getTopPages(null, null, $limit) ?? [];

            if (empty($gaRows)) {
                $notify[] = ['warning', 'No GA data available or GA not configured.'];
                return back()->withNotify($notify);
            }

            foreach ($gaRows as $r) {
                $path = $r['path'] ?? null;
                if (empty($path)) continue;
                $title = $r['title'] ?? $path;
                $views = (int) ($r['views'] ?? 0);

                $pv = PageView::where('path', $path)->first();
                if ($pv) {
                    // keep existing last_viewed_at, update title and ensure views at least GA's value
                    $pv->title = $title ?: $pv->title;
                    $pv->views = max((int) $pv->views, $views);
                    $pv->save();
                } else {
                    PageView::create([
                        'path' => $path,
                        'title' => $title,
                        'views' => $views,
                        'last_viewed_at' => null
                    ]);
                }
            }

            $notify[] = ['success', 'Imported GA top pages into page_views.'];
            return back()->withNotify($notify);
        } catch (\Exception $e) {
            \Log::error('Import GA pages failed: ' . $e->getMessage());
            $notify[] = ['error', 'Failed to import GA pages: ' . $e->getMessage()];
            return back()->withNotify($notify);
        }
    }


    public function profile()
    {
        $pageTitle = 'Profile';
        $admin = auth('admin')->user();
        return view('admin.profile', compact('pageTitle', 'admin'));
    }

    public function profileUpdate(Request $request)
    {
        $this->validate($request, [
            'name' => 'required',
            'email' => 'required|email',
            'image' => ['nullable','image',new FileTypeValidate(['jpg','jpeg','png'])]
        ]);
        $user = auth('admin')->user();

        if ($request->hasFile('image')) {
            try {
                $old = $user->image;
                $user->image = fileUploader($request->image, getFilePath('adminProfile'), getFileSize('adminProfile'), $old);
            } catch (\Exception $exp) {
                $notify[] = ['error', 'Couldn\'t upload your image'];
                return back()->withNotify($notify);
            }
        }

        $user->name = $request->name;
        $user->email = $request->email;
        $user->save();
        $notify[] = ['success', 'Profile has been updated successfully'];
        return to_route('admin.profile')->withNotify($notify);
    }


    public function password()
    {
        $pageTitle = 'Password Setting';
        $admin = auth('admin')->user();
        return view('admin.profile', compact('pageTitle', 'admin'));
    }

    public function passwordUpdate(Request $request)
    {
        $this->validate($request, [
            'old_password' => 'required',
            'password' => 'required|min:5|confirmed',
        ]);

        $user = auth('admin')->user();
        if (!Hash::check($request->old_password, $user->password)) {
            $notify[] = ['error', 'Password doesn\'t match!!'];
            return back()->withNotify($notify);
        }
        $user->password = bcrypt($request->password);
        $user->save();
        $notify[] = ['success', 'Password changed successfully.'];
        return to_route('admin.profile')->withNotify($notify);
    }

    public function notifications(){
        $notifications = AdminNotification::orderBy('id','desc')->with('user')->paginate(getPaginate());
        $pageTitle = 'Notifications';
        return view('admin.notifications',compact('pageTitle','notifications'));
    }


    public function notificationRead($id){
        $notification = AdminNotification::findOrFail($id);
        $notification->read_status = 1;
        $notification->save();
        $url = $notification->click_url;
        if ($url == '#') {
            $url = url()->previous();
        }
        return redirect($url);
    }

    public function readAll(){
        AdminNotification::where('read_status',0)->update([
            'read_status'=>1
        ]);
        $notify[] = ['success','Notifications read successfully'];
        return back()->withNotify($notify);
    }

    public function downloadAttachment($fileHash)
    {
        $filePath = decrypt($fileHash);
        $extension = pathinfo($filePath, PATHINFO_EXTENSION);
        $general = gs();
        $title = slug($general->site_name).'- attachments.'.$extension;
        $mimetype = mime_content_type($filePath);
        header('Content-Disposition: attachment; filename="' . $title);
        header("Content-Type: " . $mimetype);
        return readfile($filePath);
    }


}
