<?php

namespace App\Console\Commands;

use Illuminate\Console\Command;
use Carbon\Carbon;
use Illuminate\Support\Facades\DB;
use App\Services\GoogleAnalyticsService;

class GABackfillPageViews extends Command
{
    protected $signature = 'ga:backfill-page-views {--from=} {--to=} {--mode=aggregated}';
    protected $description = 'Backfill aggregated page view counts from Google Analytics into page_view_logs';

    public function handle()
    {
        $mode = $this->option('mode') ?? 'aggregated';
        $fromOpt = $this->option('from');
        $toOpt = $this->option('to');

        $to = $toOpt ? Carbon::parse($toOpt)->endOfDay() : Carbon::now()->endOfDay();
        $from = $fromOpt ? Carbon::parse($fromOpt)->startOfDay() : Carbon::now()->subDays(29)->startOfDay();

        $this->info("Backfill mode={$mode} from={$from->toDateString()} to={$to->toDateString()}");

        $ga = new GoogleAnalyticsService();

        $current = $from->copy();
        $totalRows = 0;
        $totalViews = 0;

        while ($current->lte($to)) {
            $dayStart = $current->copy()->startOfDay();
            $dayEnd = $current->copy()->endOfDay();
            $this->info("Processing {$dayStart->toDateString()}...");

            try {
                // Fetch top pages for this day from GA
                $rows = $ga->getTopPages($dayStart, $dayEnd, 2000) ?? [];
            } catch (\Exception $e) {
                $this->error('GA fetch failed for ' . $dayStart->toDateString() . ': ' . $e->getMessage());
                $rows = [];
            }

            // Ensure a compact daily table exists for aggregated imports
            DB::statement("CREATE TABLE IF NOT EXISTS page_view_daily (
                id BIGINT PRIMARY KEY AUTO_INCREMENT,
                path VARCHAR(191) NOT NULL,
                title VARCHAR(191),
                views BIGINT DEFAULT 0,
                view_date DATE,
                created_at TIMESTAMP NULL,
                updated_at TIMESTAMP NULL,
                UNIQUE KEY pv_daily_path_date (path, view_date)
            ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;");

            foreach ($rows as $r) {
                $path = $r['path'] ?? null;
                if (empty($path)) continue;
                // skip admin paths
                if (str_starts_with($path, '/admin')) continue;

                $title = $r['title'] ?? $path;
                $views = (int) ($r['views'] ?? 0);

                // Upsert aggregated row into page_view_daily keyed by (path, view_date)
                DB::table('page_view_daily')->updateOrInsert(
                    ['path' => $path, 'view_date' => $dayStart->toDateString()],
                    [
                        'title' => $title,
                        'views' => $views,
                        'created_at' => $dayStart->toDateTimeString(),
                        'updated_at' => now()->toDateTimeString(),
                    ]
                );

                $totalRows++;
                $totalViews += $views;
            }

            $current->addDay();
        }

        $this->info("Done. Rows upserted={$totalRows} totalViews={$totalViews}");
        return 0;
    }
}
